/**
 * \file caam_op_digest.c
 *
 * \brief Architecture specific implementation of functions relevant for digest
 *
 * \author Christoph Gellner (cgellner@de.adit-jv.com)
 *
 * \copyright (c) 2017 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 *
 ***********************************************************************/

#include <stdlib.h>
#include <sys/ioctl.h>
#include <errno.h>
#include <string.h>
#include "caam.h"
#include <linux/caam_ee.h>
#include <sdc/arch/arch_session.h>

#ifdef CAAM_EE_FLAG_INTERLEAVED_OP_BIT
    #define CAAM_EE_DGST_DEFAULT_FLAGS CAAM_EE_FLAG_INTERLEAVED_OP_BIT
#else
    #define CAAM_EE_DGST_DEFAULT_FLAGS 0
#endif

#define CAAM_EE_INIT_UPDATE_FINALIZE_FLAG 0  /* No interleaving mode is used */

/* define default type for digest in IMX6 */
static const sdc_dgst_type_t sdc_dgst_default_type =
{SDC_DGST_HASH_SHA256, CAAM_EE_HASH_SHA256_FLAG, CAAM_EE_HASH_SHA256_IDX};


/* defined in sdc_arch.h */
const sdc_dgst_type_t *sdc_arch_dgst_get_default(void)
{
    return &sdc_dgst_default_type;
}


/* defined in sdc_arch.h */
sdc_error_t sdc_arch_dgst_type_alloc(sdc_dgst_type_t **type)
{
    *type = malloc(sizeof(sdc_dgst_type_t));
    if (!(*type))
        return SDC_NO_MEM;

    /* initialize with default */
    memcpy(*type, &sdc_dgst_default_type, sizeof(sdc_dgst_type_t));

    return SDC_OK;
}

/* defined in sdc_arch.h */
sdc_error_t sdc_arch_dgst_type_free(sdc_dgst_type_t *type)
{
    free (type);

    return SDC_OK;
}

static sdc_error_t caam_map_sdc_dgst_hash (sdc_dgst_hash_t hash, uint32_t *flag, uint32_t *idx)
{
    sdc_error_t err = SDC_OK;

    switch(hash) {
    case SDC_DGST_HASH_MD5:
        *flag = CAAM_EE_HASH_MD5_FLAG;
        *idx =  CAAM_EE_HASH_MD5_IDX;
        break;
    case SDC_DGST_HASH_SHA1:
        *flag = CAAM_EE_HASH_SHA1_FLAG;
        *idx =  CAAM_EE_HASH_SHA1_IDX;
        break;
    case SDC_DGST_HASH_SHA224:
        *flag = CAAM_EE_HASH_SHA224_FLAG;
        *idx =  CAAM_EE_HASH_SHA224_IDX;
        break;
    case SDC_DGST_HASH_SHA256:
        *flag = CAAM_EE_HASH_SHA256_FLAG;
        *idx =  CAAM_EE_HASH_SHA256_IDX;
        break;
    default:
        err = SDC_ALG_MODE_INVALID;
    }

    return err;
}

/* defined in sdc_arch.h */
sdc_error_t sdc_arch_dgst_type_set_hash(sdc_dgst_type_t *type, sdc_dgst_hash_t hash)
{
    sdc_error_t err = SDC_OK;
    uint32_t hash_flag;
    uint32_t hash_idx;

    err = caam_map_sdc_dgst_hash(hash, &hash_flag, &hash_idx);

    if (err == SDC_OK) {
        type->hash = hash;
        type->arch_hash_flag = hash_flag;
        type->arch_hash_idx = hash_idx;
        err = SDC_OK;
    }

    return err;
}

/* defined in sdc_arch.h */
sdc_error_t sdc_arch_dgst_type_get_hash(const sdc_dgst_type_t *type, sdc_dgst_hash_t *hash)
{

    *hash = type->hash;

    return SDC_OK;
}

/* defined in sdc_arch.h */
sdc_error_t sdc_arch_dgst_desc_fill (
    const sdc_dgst_type_t *type,
    sdc_dgst_desc_t *desc)
{
    const struct caam_ee_hash_desc *hash_desc;
    uint32_t hash_idx;

    hash_idx = type->arch_hash_idx;
    hash_desc = &caam_ee_hash_modes_descs[hash_idx];

    desc->digest.max = hash_desc->len_max;
    desc->digest.min = hash_desc->len_min;
    desc->digest.mod = 0;
    desc->digest.dflt = desc->digest.max; /*is RFC2104 recommended min value */

    desc->supports_iuf = true;
    desc->data.block_len = hash_desc->block_size;
    desc->data.max_chunk_len = hash_desc->max_block_len;
    desc->data.chunk_len_aligned = true;
    desc->data.padding = SDC_PADDING_NO;

    if (hash_desc->final_block_aligned) {
        desc->data.padding = SDC_PADDING_PKCS7;
    }

    /* calculate max and alignment for plain and cipher */
    return sdc_intern_get_plain_cipher_spec(
               desc->data.padding,
               0,
               hash_desc->max_total_len,
               desc->data.block_len,
               &(desc->data.plain),
               NULL);
}

sdc_error_t sdc_arch_dgst(sdc_session_t *session,
                          const sdc_dgst_type_t *type,
                          const sdc_dgst_desc_t *desc,
                          const uint8_t *in_data,
                          const size_t in_data_len,
                          uint8_t *digest_out_data, const size_t digest_out_len)
{
    sdc_error_t err = SDC_OK;
    uint8_t padding_buffer[CAAM_EE_MAX_DGST_BLOCK_ALIGN];
    const uint8_t *in_data_ptr;
    size_t in_len_remaining;
    size_t block_idx;
    size_t buf_len;
    size_t align;
    size_t stop_len;
    struct caam_ee_dgst_params iodata;
    int res;
    int res_errno;
    bool padding_ioctl;

    if (desc->data.block_len > CAAM_EE_MAX_DGST_BLOCK_ALIGN) {
        /* buffer not sufficient to handle padding */
        return SDC_UNKNOWN_ERROR;
    }

    if (caam_is_op_in_progress(session)) {
        /* Cancel non-completed operation if any */
        err = caam_perform_abort_operation(session);
        if (err != SDC_OK)
            return err;
    }

    in_data_ptr = in_data;
    in_len_remaining = in_data_len;
    block_idx = 0;

    /* Initialize to no padding required
     * If no padding is required the complete data is handled by the
     * while loop only */
    stop_len = 0;
    /* padding is required in case hash needs it and not already aligned */
    if (desc->data.padding != SDC_PADDING_NO)
        stop_len = desc->data.block_len - 1;

    iodata.flags = type->arch_hash_flag | CAAM_EE_DGST_DEFAULT_FLAGS;
    iodata.dgst = digest_out_data;
    iodata.dgst_len = digest_out_len;

    /* if the first block (without padding) or if more data than padding len */
    while (((in_len_remaining > stop_len) || ((stop_len == 0) && (block_idx == 0))) && (err == SDC_OK)) {
        buf_len = in_len_remaining;

        if (buf_len > desc->data.max_chunk_len)
            buf_len = desc->data.max_chunk_len;

        in_len_remaining -= buf_len;

        iodata.flags &= ~CAAM_OP_MASK;

        if ((in_len_remaining == 0) && (stop_len == 0)) {
            /* SINGLE or MULTI_FINALIZE will only happen if no padding is required */

            /* final or init-final */
            if (block_idx == 0) {
                iodata.flags |= CAAM_OP_SINGLE;
            } else {
                iodata.flags |= CAAM_OP_MULTI_FINALIZE;
            }
        } else {
            /* init or update */

            /* alignment */
            align = buf_len % desc->data.block_len;
            buf_len -= align;
            in_len_remaining += align;

            if (block_idx == 0) {
                iodata.flags |= CAAM_OP_MULTI_INIT;
            } else {
                iodata.flags |= CAAM_OP_MULTI_UPDATE;
            }
        }

        iodata.in = (uint8_t*)in_data_ptr;
        iodata.in_len = buf_len;

        res = ioctl(session->arch.fd, CAAM_DGST_DATA, &iodata);

        if (res == 0) {
            block_idx++;
            in_data_ptr += buf_len;
        } else {
            res_errno = errno;

            /* ignore and retry after EAGAIN */
            if (res_errno != EAGAIN) {
                err = error_from_ioctl_errno(res_errno, CAAM_DGST_DATA, iodata.flags);
            }

            in_len_remaining += buf_len;
        }
    }
    if ((stop_len) && (err == SDC_OK)) {
        /* 0 <= in_len_remaining <= stop_len = block-alignment - 1 */
        if (in_len_remaining) {
            /* copy remaining_data to padding buffer */
            memcpy(padding_buffer, in_data_ptr, in_len_remaining);
        }
        err = sdc_intern_pad(SDC_PADDING_PKCS7, padding_buffer, in_len_remaining, desc->data.block_len);

        iodata.flags &= ~CAAM_OP_MASK;
        if (block_idx == 0) {
            iodata.flags |= CAAM_OP_SINGLE;
        } else {
            iodata.flags |= CAAM_OP_MULTI_FINALIZE;
        }
        iodata.in = padding_buffer;
        iodata.in_len = desc->data.block_len;

        padding_ioctl = true;
        while ((padding_ioctl) && (err == SDC_OK)) {
            res = ioctl(session->arch.fd, CAAM_DGST_DATA, &iodata);
            if (res != 0) {
                res_errno = errno;

                /* ignore and retry after EAGAIN */
                if (res_errno != EAGAIN) {
                    err = error_from_ioctl_errno(res_errno, CAAM_DGST_DATA, iodata.flags);
                }
            } else {
                padding_ioctl = false;
            }
        }
    }

    return err;
}

sdc_error_t sdc_arch_dgst_init(sdc_session_t *session,
                               const sdc_dgst_type_t *type,
                               const sdc_dgst_desc_t *desc)
{
    (void)type;
    sdc_error_t err = SDC_OK;

    if (desc->data.block_len > CAAM_EE_MAX_DGST_BLOCK_ALIGN) {
        /* buffer not sufficient to handle padding */
        return SDC_INTERNAL_ERROR;
    }

    if (caam_is_op_in_progress(session)) {
        /* Cancel non-completed operation if any */
        err = caam_perform_abort_operation(session);
        if (err != SDC_OK)
            return err;
    }

    session->arch.request = CAAM_DGST_DATA;
    session->arch.buf_idx = 0;

    return err;
}

sdc_error_t sdc_arch_dgst_update(sdc_session_t *session,
                                 const sdc_dgst_type_t *type,
                                 const sdc_dgst_desc_t *desc,
                                 const uint8_t *in_data, const size_t in_data_len)
{
    sdc_error_t err = SDC_OK;
    struct caam_ee_dgst_params iodata;
    uint32_t iodata_flags = type->arch_hash_flag | CAAM_EE_INIT_UPDATE_FINALIZE_FLAG;
    int res_errno = 0;
    int res;

    /* data needs to be aligned */
    if ((in_data_len % desc->data.block_len) != 0)
        err = SDC_INTERNAL_ERROR;

    if (err == SDC_OK) {
        if (session->arch.buf_idx == 0) {
            iodata_flags |= CAAM_OP_MULTI_INIT;
        } else {
            iodata_flags |= CAAM_OP_MULTI_UPDATE;
        }

        /* retry in case of EAGAIN */
        while (err == SDC_OK) {
            res = caam_perform_dgst_ioctl_operation(session, &iodata,
                                                    in_data, in_data_len,
                                                    NULL, 0,
                                                    iodata_flags);
            res_errno = errno;
            if (res == 0) {
                break;
            } else {
                /* ignore and retry after EAGAIN */
                if (res_errno != EAGAIN) {
                    err = error_from_ioctl_errno(res_errno, CAAM_DGST_DATA, iodata_flags);
                }
            }
        }
        if (err == SDC_OK) {
            session->arch.buf_idx++;
        }
    }

    return err;
}

sdc_error_t sdc_arch_dgst_finalize(sdc_session_t *session,
                                   const sdc_dgst_type_t *type,
                                   const sdc_dgst_desc_t *desc,
                                   const uint8_t *in_data, const size_t in_data_len,
                                   uint8_t *digest_data, size_t digest_data_len)
{
    sdc_error_t err = SDC_OK;
    struct caam_ee_dgst_params iodata;
    uint32_t iodata_flags = type->arch_hash_flag | CAAM_EE_INIT_UPDATE_FINALIZE_FLAG;
    int res_errno = 0;
    int res;
    size_t block_len = desc->data.block_len;
    const uint8_t *io_in_ptr = in_data;
    size_t io_in_len = in_data_len;

    /* in data must not exceed one block */
    if (in_data_len > block_len)
        err = SDC_INTERNAL_ERROR;

    if (err == SDC_OK) {
        if (session->arch.buf_idx == 0) {
            iodata_flags |= CAAM_OP_SINGLE;
        } else {
            iodata_flags |= CAAM_OP_MULTI_FINALIZE;
        }

        /* retry in case of EAGAIN */
        while (err == SDC_OK) {
            res = caam_perform_dgst_ioctl_operation(session, &iodata,
                                                    io_in_ptr, io_in_len,
                                                    digest_data, digest_data_len,
                                                    iodata_flags);
            res_errno = errno;
            if (res == 0) {
                break;
            } else {
                /* ignore and retry after EAGAIN */
                if (res_errno != EAGAIN) {
                    err = error_from_ioctl_errno(res_errno, CAAM_DGST_DATA, iodata_flags);
                }
            }
        }
    }

    return err;
}
